<?php
ob_start(); // Iniciar buffer de salida
require_once __DIR__ . '/../config.php'; // Incluye config.php para GEMINI_API_KEY

header('Content-Type: application/json');

try {
    // 1. VERIFICACIÓN DE SEGURIDAD Y DATOS
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido. Solo se aceptan solicitudes POST.']);
        exit;
    }

    // Verifica que la API Key esté definida y no vacía desde config.php
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada en el servidor. Verifica tu archivo config.php.']);
        exit;
    }

    $apiKey = GEMINI_API_KEY;

    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido. ' . json_last_error_msg()]);
        exit;
    }

    // Validar y sanitizar el texto, el tipo y el tono
    $adTextEs = filter_var($data['ad_text_es'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $translationType = filter_var($data['translation_type'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS); 
    $translationTone = filter_var($data['translation_tone'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

    if (empty($adTextEs)) {
        http_response_code(400);
        echo json_encode(['error' => 'No se recibió texto en español para traducir. Por favor, ingresa el texto completo.']);
        exit;
    }

    if (empty($translationType) || ($translationType !== 'Estrategica' && $translationType !== 'Literal')) { // ✅ Validar tipo de traducción
        http_response_code(400);
        echo json_encode(['error' => 'Tipo de traducción no válido. Por favor, selecciona "Estratégica" o "Literal".']);
        exit;
    }

    // El tono solo es requerido si el tipo de traducción es Estratégica
    if ($translationType === 'Estrategica' && empty($translationTone)) {
        http_response_code(400);
        echo json_encode(['error' => 'Debes seleccionar un tono de traducción si el tipo es Estratégica.']);
        exit;
    }


    // --- PROMPT MAESTRO para la IA (v3.0 con Tipo de Traducción) ---
    $prompt = "Actúa como un traductor experto y copywriter. Tu objetivo es traducir un texto del español al inglés, siguiendo las instrucciones de tipo y tono.

--- TEXTO EN ESPAÑOL A TRADUCIR ---
\"\"\"
{$adTextEs}
\"\"\"

--- TIPO DE TRADUCCIÓN SOLICITADO ---
\"\"\"
{$translationType}
\"\"\"
";

if ($translationType === 'Estrategica') {
    $prompt .= "--- TONO DE TRADUCCIÓN DESEADO (Aplicar en traducción Estratégica) ---
\"\"\"
{$translationTone}
\"\"\"
";
}

$prompt .= "
--- TU TAREA ESPECÍFICA ---
1.  Si el 'TIPO DE TRADUCCIÓN SOLICITADO' es **'Literal'**:
    * Traduce el texto **exactamente como está**, sin alteraciones, optimizaciones de marketing, ni adaptaciones. El objetivo es una fidelidad absoluta al original.
    * Genera la clave 'notas' en el JSON, pero con un array vacío o un mensaje como ['Notas no aplicables para traducción literal.'].
2.  Si el 'TIPO DE TRADUCCIÓN SOLICITADO' es **'Estrategica'**:
    * Traduce el texto publicitario completo al inglés, aplicando el 'Tono de Traducción Deseado' de forma consistente. La traducción debe ser persuasiva, centrada en ventas y optimizada para un público angloparlante.
    * Genera 2 o 3 'notas' estratégicas cortas (máximo 20 palabras cada una) **en español** que expliquen algún aspecto clave de la traducción o un consejo de marketing relacionado, considerando el tono aplicado.

--- FORMATO DE RESPUESTA REQUERIDO ---
La respuesta DEBE ser un objeto JSON válido con la siguiente estructura y claves exactas:
{
    \"traduccion\": \"[Aquí va el texto traducido al inglés]\",
    \"notas\": [
        \"[Nota estratégica 1 en español]\",
        \"[Nota estratégica 2 en español]\"
    ]
}
";

    // 3. LLAMADA A LA API DE GOOGLE GEMINI CON ESQUEMA FORZADO
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.5-flash:generateContent?key=' . $apiKey;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'traduccion' => ['type' => 'STRING'],
                    'notas' => [
                        'type' => 'ARRAY',
                        'items' => ['type' => 'STRING']
                        // Ya no se fuerza minItems/maxItems aquí, se deja a la IA decidir según el tipo.
                        // La validación en PHP verificará que sea un array de strings.
                    ]
                ],
                'required' => ['traduccion', 'notas']
            ]
        ]
    ];

    $ch = curl_init($apiUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => json_encode($payload),
    CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
    CURLOPT_TIMEOUT => 90, // Aumentar el tiempo de espera para IA
    
    // --- AÑADE ESTAS DOS LÍNEAS ---
    CURLOPT_SSL_VERIFYPEER => true, // Verificar el certificado SSL
    CURLOPT_SSL_VERIFYHOST => 2    // Asegurar que el nombre del host coincide
]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // 4. PROCESAMIENTO Y VALIDACIÓN DE LA RESPUESTA DE LA IA
    if ($response_ai === false || $http_code !== 200) {
        http_response_code($http_code === 0 ? 503 : 502); // 503 si no hay conexión, 502 si hay error de la IA
        error_log("Error de conexión/respuesta de IA en api-traductor-ingles-inteligente.php: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Raw: " . $response_ai);
        echo json_encode(['error' => 'No se pudo obtener una respuesta válida del servicio de IA.', 'details' => $curl_error ?: 'No hay detalles de cURL.']);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;

    if ($response_text === null) {
        http_response_code(500);
        error_log("La respuesta de la IA no contenía el formato esperado (partes de texto) en api-traductor-ingles-inteligente.php. Raw: " . $response_ai);
        echo json_encode(['error' => 'La respuesta de la IA no contenía el formato esperado (partes de texto).', 'raw_response' => $response_ai]);
        exit;
    }

    // Limpiar el texto de posibles bloques de código Markdown
    $response_text = str_replace(['```json', '```'], '', $response_text);

    $final_data = json_decode(trim($response_text), true);

    if ($final_data === null || !isset($final_data['traduccion']) || !isset($final_data['notas']) || !is_array($final_data['notas']) || json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(500);
        error_log("La IA devolvió un formato de JSON inválido o la estructura de datos esperada no está completa/correcta en api-traductor-ingles-inteligente.php. Raw AI Text: " . trim($response_text) . ", JSON Error: " . json_last_error_msg());
        echo json_encode([
            'error' => 'La IA devolvió un formato de JSON inválido o la estructura de datos esperada no está completa/correcta. Intenta de nuevo con tus datos.',
            'raw_ai_text' => trim($response_text),
            'json_error_message' => json_last_error_msg(),
            'decoded_data_debug' => $final_data
        ]);
        exit;
    }

    // Si todo es correcto, enviar la respuesta validada.
    ob_end_clean(); // Limpiar el buffer si todo es correcto y listo para la salida
    echo json_encode($final_data);
    exit;

} catch (Exception $e) {
    // Capturar cualquier excepción inesperada y devolver un JSON de error.
    http_response_code(500);
    error_log("Excepción interna en api-traductor-ingles-inteligente.php: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    // Asegurarse de que el búfer de salida se detenga y limpie al final.
    while (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>