<?php
// /IAMarketing/api/api-services-descriptions.php
ob_start();
require_once __DIR__ . '/../config.php';
ob_clean();

header('Content-Type: application/json');

try {
    // 1. VERIFICACIÓN DE SEGURIDAD Y CONSTANTES
    // Asumimos que GEMINI_API_KEY y GEMINI_MODEL están definidos en config.php
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini (GEMINI_API_KEY) no está configurada.']);
        exit;
    }
    if (!defined('GEMINI_MODEL') || empty(GEMINI_MODEL)) {
        http_response_code(500);
        echo json_encode(['error' => 'El modelo de Gemini (GEMINI_MODEL) no está configurado.']);
        exit;
    }
    
    $apiKey = GEMINI_API_KEY;

    // Verificación del método HTTP
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Método no permitido.']);
        exit;
    }

    // Obtención y validación de datos
    $json_data = file_get_contents('php://input');
    $data = json_decode($json_data, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido.']);
        exit;
    }

    // Sanitizar y validar las entradas
    $service_name = filter_var($data['service_name'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $service_features = filter_var($data['service_features'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $value_proposition = filter_var($data['value_proposition'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $problem_solved = filter_var($data['problem_solved'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $key_benefit = filter_var($data['key_benefit'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $target_audience = filter_var($data['target_audience'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $additional_details = filter_var($data['additional_details'] ?? '', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

    // Validar campos obligatorios
    if (empty($service_name) || empty($service_features) || empty($problem_solved) || empty($key_benefit)) {
        http_response_code(400);
        echo json_encode(['error' => 'Faltan campos obligatorios.']);
        exit;
    }

    // Construcción del PROMPT MAESTRO (Incluye viñetas y CTA forzados)
    $prompt = "Actúa como un experto en redacción de textos publicitarios para negocios de servicios locales. Tu objetivo es tomar un servicio genérico y convertirlo en descripciones persuasivas que se centren en los beneficios para el cliente.

    **CONTEXTO DEL SERVICIO:**
    - Nombre del Servicio: \"{$service_name}\"
    - Propuesta de Valor Estratégica: \"{$value_proposition}\"
    - Qué Incluye el Servicio: \"{$service_features}\"
    - Problema que Resuelve: \"{$problem_solved}\"
    - Beneficio Clave: \"{$key_benefit}\"
    " . (!empty($target_audience) ? "- Público Objetivo: \"{$target_audience}\"\n" : "") . "
    " . (!empty($additional_details) ? "- Detalles Adicionales: \"{$additional_details}\"\n" : "") . "

    **TU TAREA:**
    Genera 2 versiones de descripciones del servicio. Cada versión debe tener un enfoque diferente, optimizado para un canal de comunicación distinto (una corta para anuncios y una detallada para web).

    Para cada versión, genera lo siguiente:
    1.  **type:** El tipo de descripción (ej. 'Descripción Corta', 'Descripción Detallada').
    2.  **description:** Una breve explicación (1-2 oraciones) del propósito de esa descripción.
    3.  **text_es:** La descripción completa en español. Debe ser persuasiva, enfocada en el beneficio y usar un lenguaje simple. **DEBE incluir VIÑETAS y un CTA si aplica al tipo de descripción.**
    4.  **text_en:** La traducción precisa del `text_es` al inglés.

    **REGLAS IMPORTANTES:**
    -   **El texto de la Descripción Detallada DEBE contener viñetas** (`-` o `*` seguidas de `\n`) para las inclusiones o beneficios.
    -   **NO uses marcadores de posición** como `[Nombre del Servicio]` en el resultado final.
    -   **NO incluyas texto introductorio o explicaciones** fuera del objeto JSON.
    -   La salida debe ser un **objeto JSON válido** que contenga un array llamado `descriptions`. Este array debe contener 2 objetos, cada uno con las claves \"type\", \"description\", \"text_es\" y \"text_en\".
    ";

    // 2. CONSTRUCCIÓN DE LA URL CON CONSTANTES
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . GEMINI_API_KEY;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'descriptions' => [
                        'type' => 'ARRAY',
                        'items' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'type' => ['type' => 'STRING'],
                                'description' => ['type' => 'STRING'],
                                'text_es' => ['type' => 'STRING'],
                                'text_en' => ['type' => 'STRING']
                            ],
                            'required' => ['type', 'description', 'text_es', 'text_en']
                        ],
                        'minItems' => 2,
                        'maxItems' => 2
                    ]
                ],
                'required' => ['descriptions']
            ]
        ]
    ];

    // 3. EJECUCIÓN DE cURL
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);

    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    // 4. PROCESAMIENTO Y VALIDACIÓN
    if ($response_ai === false || $http_code !== 200) {
        http_response_code($http_code === 0 ? 503 : 502);
        $error_details = $curl_error ?: ($response_ai ? json_decode($response_ai, true) : 'Respuesta de la API vacía o ilegible.');
        error_log("Error de conexión/respuesta de IA: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Details: " . json_encode($error_details, JSON_UNESCAPED_UNICODE));
        echo json_encode(['error' => 'No se pudo obtener una respuesta válida del servicio de IA.', 'details' => $error_details, 'http_code' => $http_code]);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    $response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? null;
    
    if ($response_text === null) {
        http_response_code(500);
        error_log("La respuesta de la IA no contenía el formato esperado. Raw: " . json_encode($result_array, JSON_UNESCAPED_UNICODE));
        echo json_encode(['error' => 'La IA devolvió un formato de respuesta inesperado.']);
        exit;
    }

    $response_text = str_replace(['```json', '```'], '', $response_text);
    $final_data = json_decode(trim($response_text), true);

    if ($final_data === null || !isset($final_data['descriptions']) || !is_array($final_data['descriptions']) || count($final_data['descriptions']) !== 2) {
        http_response_code(500);
        error_log("La IA devolvió un formato de JSON inválido. Raw AI Text: " . trim($response_text));
        echo json_encode(['error' => 'La IA devolvió un formato de JSON inválido.']);
        exit;
    }

    // 5. ENVÍO DE LA RESPUESTA FINAL
    ob_end_clean();
    echo json_encode($final_data, JSON_UNESCAPED_UNICODE);
    exit;

} catch (Exception $e) {
    http_response_code(500);
    error_log("Excepción interna: " . $e->getMessage());
    echo json_encode(['error' => 'Error interno del servidor: ' . $e->getMessage()]);
    exit;
} finally {
    if (ob_get_level() > 0) {
        ob_end_flush();
    }
}
?>