<?php
ob_start();
require_once __DIR__ . '/../files/guardiankey.php';
require_once __DIR__ . '/../config.php';

$apiKey = GEMINI_API_KEY;

// Asumimos que GEMINI_MODEL está definido en /config.php (ej: 'gemini-2.5-flash')
if (!defined('GEMINI_MODEL') || empty(GEMINI_MODEL)) {
    // Si GEMINI_MODEL no está definida, se asume un valor por defecto seguro (igual al del archivo original, si se quiere)
    // Sin embargo, para mantener la coherencia con su solicitud, no definimos aquí un valor.
    error_log("Error: La constante GEMINI_MODEL no está definida en config.php.");
    http_response_code(500);
    echo json_encode(['error' => 'La constante GEMINI_MODEL no está configurada en la aplicación.']);
    exit;
}

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!defined('GEMINI_API_KEY') || empty(GEMINI_API_KEY)) {
        http_response_code(500);
        echo json_encode(['error' => 'La clave de API de Gemini no está configurada.']);
        exit;
    }

    $data = json_decode(file_get_contents('php://input'), true);

    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("Error al decodificar JSON en la API: " . json_last_error_msg());
        http_response_code(400);
        echo json_encode(['error' => 'Formato de datos JSON inválido.']);
        exit;
    }

    // **********************************************
    // CORRECCIÓN: Extraer la Propuesta de Valor de $data
    // **********************************************
    $business_name = $data['business_name'] ?? '';
    $phone_whatsapp = $data['phone_whatsapp'] ?? '';
    $website_url = $data['website_url'] ?? '';
    $service_zone = $data['service_zone'] ?? '';
    
    // CORREGIDO: Propuesta de valor extraída del input oculto del formulario
    $value_proposition = $data['value_proposition'] ?? ''; 

    // Campos consolidados
    $product_service_description_simplified = $data['product_service_description_simplified'] ?? '';
    $client_description_simplified = $data['client_description_simplified'] ?? '';

    // Limpiar la URL de 'http://' o 'https://' para un CTA más limpio
    if (!empty($website_url)) {
        $website_url_for_cta = preg_replace('/^https?:\/\//', '', $website_url);
    } else {
        $website_url_for_cta = '';
    }

    // --- PROMPT MAESTRO (v1.0 - Creador de Publicaciones Sociales Bilingüe con Hashtags) ---
    $prompt = "Actúa como un experto en marketing digital y copywriter para negocios locales. Tu tarea es generar publicaciones atractivas para redes sociales que capten la atención y generen interacción, siempre en español y en inglés, incluyendo hashtags relevantes.

--- FILOSOFÍA CLIENT3 ---
- Los posts deben ser concisos y enfocados.
- Deben seguir la estructura de Problema -> Solución -> Servicio.
- Utilizar un tono de comunicación consistente con la marca.
- Incluir un llamado a la acción claro y fácil de seguir.
- Tono de Comunicación Predeterminado: Amigable

--- TU TAREA ESPECÍFICA ---
Genera 3 tipos de publicaciones, cada una con su versión en español y en inglés, y con sus respectivos hashtags.

--- DATOS DEL NEGOCIO ---
- Nombre del Negocio: {$business_name}
- Página Web/Perfil Social: {$website_url_for_cta}
- Teléfono/WhatsApp: {$phone_whatsapp}
- Zona de Servicio: {$service_zone}

--- CONTEXTO CLAVE DE MARCA ---
- Propuesta de Valor (Base Estratégica): {$value_proposition}

--- DESCRIPCIÓN CONSOLIDADA ---
- Descripción del Producto/Servicio Simplificada: {$product_service_description_simplified}
- Descripción del Cliente Simplificada: {$client_description_simplified}

--- INSTRUCCIONES PARA LA IA ---
Basado en la 'Propuesta de Valor', la 'Descripción del Producto/Servicio' y la 'Descripción del Cliente', crea los textos:
    - ¿Qué es el producto/servicio?
    - ¿Qué problemas específicos resuelve?
    - ¿Cuáles son los beneficios clave?

Utiliza toda esta información extraída para generar las publicaciones bajo las siguientes estructuras:

--- ESTRUCTURA DE LAS PUBLICACIONES ---
1.  **Publicación Enfocada en el Problema:**
    -   **Copy:** Un texto corto que agita el problema del cliente (extraído de la 'Descripción del Cliente'), presenta brevemente el servicio como la solución (alineado con la 'Propuesta de Valor') y termina con un CTA claro que incluya el teléfono y la web (si aplica).
    -   **Hashtags:** Relevantes al problema, el servicio y la zona (si aplica).
2.  **Publicación Enfocada en la Solución/Deseo:**
    -   **Copy:** Un texto que pinta el resultado ideal que el cliente desea (extraído de la 'Descripción del Cliente'), muestra cómo tu servicio lo logra (alineado con la 'Propuesta de Valor') y termina con un CTA claro que incluya el teléfono y la web (si aplica).
    -   **Hashtags:** Relevantes al deseo/solución, el servicio y la zona (si aplica).
3.  **Publicación Enfocada en el Servicio/Beneficios:**
    -   **Copy:** Describe el servicio y sus beneficios clave (extraídos de la 'Descripción del Producto') usando emojis amigables (ej. ✅ o ⭐) y termina con un CTA claro que incluya el teléfono y la web (si aplica).
    -   **Hashtags:** Relacionados con el servicio, sus beneficios y la zona (si aplica).

--- REGLAS ESTRICTAS PARA LA GENERACIÓN ---
-   **Output Bilingüe:** Para CADA publicación, DEBES proporcionar una versión en 'es' (español) y 'en' (inglés).
-   **Hashtags:** Cada versión (es/en) debe tener su propio array de 5 a 10 hashtags. Los hashtags NO deben incluir espacios y deben ser relevantes. Los hashtags geográficos solo si 'Zona de Servicio' se proporciona.
-   **Concisión:** El copy de cada publicación (es/en) debe ser conciso, ideal para redes sociales (máximo 5-7 líneas).
-   **Llamado a la Acción (CTA):** En cada copy (es/en), el CTA DEBE incluir el 'Teléfono/WhatsApp' {$phone_whatsapp}. Si 'Página Web/Perfil Social' {$website_url_for_cta} se proporciona, también debe incluirse. El CTA debe ser claro (ej. 'Contáctanos hoy', 'Envíanos un mensaje', 'Visita nuestra web' o 'Visita {$website_url_for_cta}' - **ASEGÚRATE de NO incluir 'http://' o 'https://' en las URLs en el CTA**).
-   **Tono:** El tono de la publicación debe ser coherentemente amigable.
-   **Emojis:** Usa emojis de forma profesional y moderada si encajan con el tono (ej. ✨, 🚀, ✅, ⭐).

--- FORMATO DE RESPUESTA REQUERIDO ---
La respuesta DEBE ser un único objeto JSON válido con una clave 'posts', que es un array de objetos. Cada objeto en 'posts' debe tener:
- 'type': ('problem', 'solution', 'service')
- 'es_copy': string
- 'es_hashtags': array de strings
- 'en_copy': string
- 'en_hashtags': array de strings
";

    // 3. LLAMADA A LA API CON ESQUEMA FORZADO (Y MODELO CENTRALIZADO)
    $apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/' . GEMINI_MODEL . ':generateContent?key=' . $apiKey;

    $payload = [
        'contents' => [['parts' => [['text' => $prompt]]]],
        'generationConfig' => [
            'response_mime_type' => 'application/json',
            'response_schema' => [
                'type' => 'OBJECT',
                'properties' => [
                    'posts' => [
                        'type' => 'ARRAY',
                        'items' => [
                            'type' => 'OBJECT',
                            'properties' => [
                                'type' => ['type' => 'STRING'],
                                'es_copy' => ['type' => 'STRING'],
                                'es_hashtags' => ['type' => 'ARRAY', 'items' => ['type' => 'STRING']],
                                'en_copy' => ['type' => 'STRING'],
                                'en_hashtags' => ['type' => 'ARRAY', 'items' => ['type' => 'STRING']]
                            ],
                            'required' => ['type', 'es_copy', 'es_hashtags', 'en_copy', 'en_hashtags']
                        ]
                    ]
                ],
                'required' => ['posts']
            ]
        ]
    ];
    
    $ch = curl_init($apiUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => json_encode($payload),
        CURLOPT_HTTPHEADER => ['Content-Type: application/json'],
        CURLOPT_TIMEOUT => 90
    ]);
    $response_ai = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($response_ai === false || $http_code !== 200) {
        http_response_code(502);
        $error_details = $curl_error ?: ($response_ai ? json_decode($response_ai, true) : 'Respuesta vacía o ilegible de la IA.');
        error_log("Error de IA en la API: HTTP Code " . $http_code . ", cURL Error: " . $curl_error . ", Details: " . json_encode($error_details));
        echo json_encode([
            'error' => 'No se pudo obtener una respuesta válida del servicio de IA. Posiblemente un problema con la API Key o el formato de la solicitud.',
            'http_code' => $http_code,
            'curl_error' => $curl_error,
            'raw_response_from_ai' => $response_ai,
            'details' => $error_details
        ]);
        exit;
    }

    $result_array = json_decode($response_ai, true);
    
    $raw_response_text = $result_array['candidates'][0]['content']['parts'][0]['text'] ?? '';
    
    // Elimina el marcado de código JSON que a veces añade Gemini
    $cleaned_json_text = str_replace(['```json', '```'], '', $raw_response_text);
    $final_data = json_decode(trim($cleaned_json_text), true);

    if ($final_data === null || !isset($final_data['posts']) || !is_array($final_data['posts']) || empty($final_data['posts'])) {
        http_response_code(500);
        error_log("Formato de respuesta de IA inválido en la API. Respuesta cruda: " . trim($raw_response_text));
        echo json_encode([
            'error' => 'La IA devolvió un formato de JSON inválido o la estructura de datos esperada no está completa/correcta. Intenta de nuevo con tus datos.',
            'raw_ai_response_after_clean_and_trim' => trim($cleaned_json_text),
            'decoded_data_debug' => $final_data
        ]);
        exit;
    }

    ob_end_clean();
    // Se devuelve el objeto JSON que contiene el array 'posts'
    echo json_encode($final_data);
    exit;
}
?>