<?php
// CARGA MAESTRA DE CONFIGURACIÓN Y BRANDING
require_once(__DIR__ . '/core/init.php');
require_once 'ia-marketing-results.php';
require_once 'media/videos-ia.php';

$current_radar = null;

if (isset($pdo)) {
    try {
        $sql = "SELECT puntuacion, diagnostico_general, plan_de_accion, situations FROM ia_business_diagnostic_radar_results WHERE business_info_id = ? LIMIT 1";
        $stmt = $pdo->prepare($sql);
        // Implementación de Sentencias Preparadas (Mitigación SQLi)
        $stmt->execute([$business_id]); 
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            // Decodificar JSONs para usar en PHP y pasar a JS
            $row['plan_de_accion'] = json_decode($row['plan_de_accion'], true);
            // Esto ahora leerá el campo 'situations' que debes añadir a la BD
            $row['situations'] = json_decode($row['situations'] ?? '[]', true); 
            $current_radar = $row;
        }
    } catch (PDOException $e) {
        error_log("Error fetching radar: " . $e->getMessage());
    }
}
// --- FIN: LÓGICA DE RECUPERACIÓN ---

$client_name = $client_info['name'] ?? '';
$business_name = $client_info['business_name'] ?? '';
$phone = $client_info['phone'] ?? '';
$whatsapp = $client_info['whatsapp'] ?? '';
$email = $client_info['email'] ?? '';
$website = $client_info['website'] ?? '';
$operating_cities = $client_info['operating_cities'] ?? '';
$logo_url = $client_info['logo_url'] ?? '';
$value_proposition = $ia_results['value_proposition'] ?? '';
$current_client_profiles = $ia_results['ideal_client_profiles'] ?? [];
$social_links = $client_info['social_links'] ?? [];

if (ob_get_level() > 0) { ob_end_clean(); }
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Radar De Diagnóstico Empresarial Con <?php echo htmlspecialchars($branding['iamarketing_name']); ?> Powered_by <?php echo htmlspecialchars($branding['powered_by']); ?></title>
    <meta name="robots" content="noindex, nofollow">
    
    <link rel="icon" type="image/png" href="<?php echo htmlspecialchars($branding['favicon']); ?>">
    <link rel="apple-touch-icon" href="<?php echo htmlspecialchars($branding['favicon']); ?>">
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="<?php echo htmlspecialchars($google_font_url); ?>">

    <?php include 'files/gtm-head.php'; ?>
    
    <script src="https://unpkg.com/lucide@latest"></script>
    <script src="https://unpkg.com/imask"></script>
    
    <link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
    <script src="https://cdn.quilljs.com/1.3.6/quill.js"></script>

    <link rel="stylesheet" href="style.css">
    <link rel="stylesheet" href="tools.css">
    <script src="files/header-manager.js"></script>

    </head>
<body data-page-title="Radar De Diagnóstico Empresarial"
      data-page-subtitle="Una Evaluación Estratégica 360° Del Estado Actual De Tu Negocio"
      data-page-icon="brain">

    <div id="toast-container" class="toast-container"></div>

    <?php include 'files/gtm-body.php'; ?>

<div class="relative min-h-screen md:flex">

    <div id="sidebar-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-30 hidden md:hidden"></div>
    <div id="task-panel-overlay" class="off-canvas-overlay"></div> 
    
    <?php include 'ia-menu.php'; ?>

    <main class="flex-1 overflow-y-auto">
        <header class="bg-white shadow-sm p-4 flex justify-between items-center sticky top-0 z-20">
            <button id="mobile-menu-button" class="md:hidden text-gray-600 hover:text-gray-800">
                <i data-lucide="menu" class="w-6 h-6"></i>
            </button>
            <div class="page-header-container">
                <h2 id="page-title"></h2>
                <p id="page-subtitle"></p>
            </div>
        </header>
        
        <div id="content-area" class="p-4 md:p-8">
            <section id="diagnostic-tool" class="dashboard-section">
                <div class="grid grid-cols-1 lg:grid-cols-5 gap-8">
                    <div class="lg:col-span-3">
                        <form id="diagnostic-form" class="space-y-8 bg-white p-8 rounded-xl shadow-md">
                            <div>
                                <div class="step-header">
                                        <h2 class="text-2xl font-black pb-0 mb-0 flex items-center uppercase">
                                            <span class="step-number text-[var(--color-primary)]"> <i data-lucide="lightbulb" class="mr-3 text-[var(--color-secondary)]"></i> PASO 1:</span>
                                            <span class="step-title ml-2"> LOS PUNTOS DÉBILES DE TU NEGOCIO
                                            </span>
                                        </h2>
                                        <p class="step-subtitle uppercase text-gray-500 font-bold mt-2">Marca todos los desafíos que enfrenta tu negocio actualmente.</p>
                                    </div>
                                <div class="border-b border-gray-200 pb-4 mb-6"></div>
                                
                                <div id="diagnostic-options" class="grid grid-cols-1 md:grid-cols-2 gap-6"></div>
                            </div>
                              <input type="hidden" name="value_proposition" value="<?php echo htmlspecialchars($value_proposition); ?>">
                        </form>
                    </div>
                    <div class="lg:col-span-2 space-y-8">
                        
                        <?php if (!empty($logo_url)): ?>
                            <div class="bg-white p-4 rounded-xl shadow-md">
                                <img src="<?php echo htmlspecialchars($logo_url); ?>" 
                                     alt="Logotipo de <?php echo htmlspecialchars($business_name); ?>" 
                                     class="block mx-auto object-contain h-24" 
                                     style="max-width: 250px;">
                            </div>
                        <?php endif; ?>
                        
                        <button id="open-radar-panel-btn" class="btn btn-primary w-full py-4 px-6 rounded-lg font-black text-lg uppercase gap-2 hover:scale-102 transition-transform shadow-lg">
                            <i data-lucide="save" class="w-5 h-5"></i> <span>Tu Radar De Diagnóstico</span>
                        </button>

                        <div class="w-full">
                                <div class="bg-black/40 rounded-xl ring-1 ring-white/10 overflow-hidden">
                                    <div class="relative w-full pb-[56.25%]">
                                        <video class="absolute top-0 left-0 w-full h-full object-cover" id="videoPlayer" controls controlslist="nodownload fullscreen" preload="metadata">
                                            <source src="https://samplelib.com/lib/preview/mp4/sample-5s.mp4" type="video/mp4">
                                            TU NAVEGADOR NO SOPORTA LA ETIQUETA DE VIDEO
                                        </video>
                                        <div id="playOverlay" class="absolute inset-0 flex justify-center items-center cursor-pointer group">
                                            <img src="img/VideoLeccion-IAMarketing.jpg" alt="VideoLeccion - IAMarketing" class="absolute inset-0 w-full h-full object-cover">
                                            <div class="absolute inset-0 w-full h-full bg-black/40 transition-colors group-hover:bg-black/60"></div>
                                            <i data-lucide="play" class="relative w-16 h-16 text-white opacity-80 group-hover:opacity-100 transition-opacity"></i>
                                        </div>
                                    </div>
                                    <div class="p-4 text-white/90 text-sm flex items-center justify-between">
                                        <span class="inline-flex items-center gap-2">
                                            <i data-lucide="play-circle" class="w-4 h-4"></i> Video Tutorial
                                        </span>
                                        <span class="inline-flex items-center gap-2">
                                            <i data-lucide="clock" class="w-4 h-4"></i> 2:30 min
                                        </span>
                                    </div>
                                </div>
                            </div>
                        <div class="bg-white text-brand-dark p-6 md:p-8 rounded-xl shadow-md">
                            <div class="step-header">
                                <h2 class="text-2xl font-black pb-0 mb-0 flex items-center uppercase justify-center md:justify-start">
                                    <i data-lucide="book-open" class="mr-3 text-[var(--color-highlight)]"></i>
                                    <span class="step-title">INSTRUCCIONES IMPORTANTES</span>
                                </h2>
                                <p class="step-subtitle text-center md:text-left uppercase text-gray-500 font-bold mt-2">REVISA ESTOS PUNTOS DETALLADAMENTE</p>
                            </div>
                            <div class="border-b border-gray-200 pb-4 mb-6"></div>
                            <ul class="space-y-4 text-gray-700">
                                <li class="flex items-start gap-3"><span class="bg-[var(--color-highlight)] p-1 rounded text-[var(--color-primary)]"><i data-lucide="list-checks" class="w-4 h-4"></i></span><div><strong class="text-[var(--color-primary)] font-black">ELIGE TUS DESAFÍOS:</strong> Marca los puntos que más te cuestan en tu negocio.</div></li>
                                <li class="flex items-start gap-3"><span class="bg-[var(--color-highlight)] p-1 rounded text-[var(--color-primary)]"><i data-lucide="message-circle-question" class="w-4 h-4"></i></span><div><strong class="text-[var(--color-primary)] font-black">AÑADE CONTEXTO:</strong> Describe tu negocio brevemente para un diagnóstico más preciso.</div></li>
                                <li class="flex items-start gap-3"><span class="bg-[var(--color-highlight)] p-1 rounded text-[var(--color-primary)]"><i data-lucide="arrow-right-to-line" class="w-4 h-4"></i></span><div><strong class="text-[var(--color-primary)] font-black">RECIBE UN PLAN:</strong> Obtén un diagnóstico claro y un plan de acción priorizado para avanzar.</div></li>
                            </ul>
                        </div>
                    </div>
                    <div class="lg:col-span-5 bg-white p-6 md:p-8 rounded-xl shadow-md mt-6">
                        <div class="step-header">
                            <h2 class="text-2xl font-black pb-0 mb-0 flex items-center uppercase">
                                <i data-lucide="pencil-ruler" class="mr-3 text-[var(--color-secondary)]"></i>
                                <span class="text-[var(--color-secondary)] mr-2">PASO 2:</span> AFINA LOS RESULTADOS
                            </h2>
                            <p class="step-subtitle uppercase text-gray-500 font-bold mt-2">Cuéntanos a qué te dedicas para recibir recomendaciones específicas para tu industria.</p>
                        </div>
                        <div class="border-b border-gray-200 pb-4 mb-6"></div>
                        <input id="diag-other" name="business_description" type="text" form="diagnostic-form" placeholder="Ej: Soy Un Contratista De Remodelación En Miami" class="w-full p-3 bg-white border border-gray-300 rounded-lg focus:ring-1 focus:ring-[var(--color-highlight)] outline-none transition-all">
                    </div>
                    <div class="lg:col-span-5 bg-white p-6 md:p-8 rounded-xl shadow-md mt-6">
                        <div class="step-header">
                            <h2 class="text-2xl font-black pb-0 mb-0 flex items-center uppercase">
                                <i data-lucide="gauge" class="mr-3 text-[var(--color-secondary)]"></i>
                                <span class="text-[var(--color-secondary)] mr-2">PASO 3:</span> GENERA TU DIAGNÓSTICO ESTRATÉGICO
                            </h2>
                            <p class="step-subtitle uppercase text-gray-500 font-bold mt-2">Descubre dónde estás y cuál es el camino ideal para crecer con intención.</p>
                        </div>
                        <div class="border-b border-gray-200 pb-4 mb-6"></div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 md:gap-8 items-center">

                            <div class="flex flex-col justify-center items-center bg-[var(--color-primary)] p-6 rounded-xl shadow-lg">
                                <i data-lucide="brain" class="w-24 h-24 md:w-32 md:h-32 text-white animate-pulse"></i>
                                <p class="mt-4 text-white font-black text-xl uppercase tracking-wide">
                                    IA<span class="text-[var(--color-highlight)]">Marketing</span>™
                                </p>
                            </div>

                            <div class="md:col-span-2 text-center md:text-left">
                                
                                <h3 class="text-2xl font-black text-gray-800 flex items-center justify-center md:justify-start mb-2 uppercase">
                                    <i data-lucide="bell-electric" class="w-6 h-6 text-[var(--color-secondary)] mr-2 flex-shrink-0"></i>
                                    Información Importante
                                </h3>
                                
                                <p class="text-lg text-gray-700 mb-6">
                                    Para una evaluación completa y precisa de tu negocio, <b>es importante</b> que aportes información de las diferentes áreas de tu negocio. Con la información que proporciones, nuestra <b>inteligencia artificial</b> generará un <b>Radar De Diagnóstico Empresarial</b>, para que obtengas una evaluación estratégica de 360° del estado actual de tu negocio
                                </p>
                                
                                <div class="flex justify-center md:justify-start">
                                    <button id="generateBtn" type="submit" form="diagnostic-form" class="btn btn-primary w-full sm:w-auto py-4 px-6 rounded-lg text-lg flex items-center justify-center uppercase shadow-xl hover:scale-105 transition-transform">
                                        
                                        <i data-lucide="arrow-big-right-dash" class="w-6 h-6"></i>
                                        
                                        <span id="btn-text">GENERAR TU RADAR DE DIAGNÓSTICO CON IA</span>
                                        
                                        <div id="loader-container" class="hidden ml-3"></div>
                                    </button>
                                </div>
                            </div>
                            
                        </div>
                        
                    </div>
                    <div id="result-container" class="lg:col-span-5 mt-8"></div>
                </div>
            </section>
        </div>
    </main>
</div>

<div id="radar-panel-overlay" class="fixed inset-0 bg-black bg-opacity-50 z-40 hidden" aria-hidden="true"></div>

<div id="radar-panel" class="fixed top-0 right-0 h-full w-full lg:w-1/2 bg-[var(--color-background)] z-50 transform translate-x-full transition-transform duration-300 ease-in-out shadow-2xl flex flex-col">
    
    <div class="flex-shrink-0 flex justify-between items-center p-4 border-b border-gray-200 bg-[var(--color-primary)] text-white shadow z-20">
        <h3 class="text-3xl font-black text-[var(--color-highlight)] flex items-center uppercase">
            <i data-lucide="radar" class="w-8 h-8 mr-2 text-white"></i>
            TU RADAR DE DIAGNÓSTICO
        </h3>
        <button id="close-radar-panel-btn" class="text-white hover:text-[var(--color-highlight)] p-1 rounded-md transition-colors duration-200">
            <i data-lucide="x" class="w-8 h-8"></i>
        </button>
    </div>

    <div class="flex-grow overflow-y-auto p-6 relative">
        
        <div id="radar-panel-content" class="bg-white p-6 rounded-xl shadow-md transition duration-300 min-h-[200px]">
            <div class="flex flex-col items-center justify-center h-40 text-gray-400">
                <i data-lucide="loader-2" class="w-8 h-8 animate-spin mb-2"></i>
                <span class="text-sm font-bold uppercase">Cargando datos del radar...</span>
            </div>
        </div>
        
    </div>

    <div class="p-4 bg-gray-100 border-t border-gray-200 flex-shrink-0 z-10"> 
        <button type="button" onclick="document.getElementById('close-radar-panel-btn').click()" class="w-full btn btn-secondary uppercase flex items-center justify-center gap-2">
            <i data-lucide="x-circle" class="w-5 h-5"></i> CERRAR PANEL DE RESULTADOS
        </button>
    </div>

</div>

<div id="confirmation-modal" 
     class="fixed inset-0 bg-gray-900 bg-opacity-90 flex items-center justify-center hidden z-[100]"
     onclick="if(event.target === this) closeModal()">
    
    <div class="bg-white rounded-xl shadow-2xl w-full max-w-sm m-4 transform transition-all duration-300 scale-95 opacity-0 text-center">
        
        <div class="modal-header-container rounded-t-xl pt-6">
            <h3 id="modal-title" class="text-4xl font-black uppercase text-[var(--color-highlight)] leading-none">CONFIRMAR</h3>
        </div>

        <div class="p-8">
            <div class="flex justify-center mb-6">
                <i id="modal-icon" data-lucide="alert-triangle" class="w-16 h-16 text-[var(--color-secondary)]"></i>
            </div>
            
            <p id="modal-message" class="text-[var(--color-primary)] mb-6 uppercase text-lg font-bold leading-tight">
                ¿ESTÁS SEGURO DE ESTA ACCIÓN?
            </p>

            <div class="flex flex-col sm:flex-row justify-center space-y-2 sm:space-y-0 sm:space-x-4 mt-8">
                <button type="button" class="btn btn-secondary font-black py-2.5 px-4 rounded-lg uppercase w-full sm:w-auto flex items-center justify-center gap-2" id="modal-cancel-btn">
                    <i data-lucide="x-circle" class="w-5 h-5"></i> CANCELAR
                </button>
                
                <button type="button" class="btn btn-primary font-black py-2.5 px-4 rounded-lg uppercase w-full sm:w-auto flex items-center justify-center gap-2 shadow-md transition-all duration-200" id="modal-confirm-btn">
                    CONFIRMAR
                </button>
            </div>
        </div>
    </div>
</div>

<script type="module">
document.addEventListener('DOMContentLoaded', function() {
    lucide.createIcons();
    
    // MENU MOBILE
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const sidebar = document.getElementById('sidebar');
    const sidebarOverlay = document.getElementById('sidebar-overlay');

    mobileMenuButton.addEventListener('click', () => {
        sidebar.classList.toggle('-translate-x-full');
        sidebarOverlay.classList.toggle('hidden');
    });

    sidebarOverlay.addEventListener('click', () => {
        sidebar.classList.add('-translate-x-full');
        sidebarOverlay.classList.add('hidden');
    });
    
    // Video Player
    const playOverlay = document.getElementById('playOverlay');
    const videoPlayer = document.getElementById('videoPlayer');
    if (playOverlay && videoPlayer) {
        videoPlayer.controls = false;
        playOverlay.addEventListener('click', () => {
            playOverlay.style.display = 'none';
            videoPlayer.controls = true;
            videoPlayer.play().catch(e => {
                if (typeof showToast === 'function') {
                    showToast("Error al reproducir video: " + e.message, 'error');
                }
            });
        });
        videoPlayer.addEventListener('ended', () => {
            playOverlay.style.display = 'flex';
            videoPlayer.controls = false;
        });
    }

    // --- DATOS GLOBALES ---
    const form = document.getElementById('diagnostic-form');
    const resultContainer = document.getElementById('result-container');
    const generateBtn = document.getElementById('generateBtn');
    // PHP inyecta los datos guardados
    let radarGuardadoActual = <?php echo json_encode($current_radar ?? null); ?>;
    let radarGeneradoReciente = null; // Almacena el último resultado de la IA

    // --- PANEL LATERAL (NUEVO) ---
    const openRadarPanelBtn = document.getElementById('open-radar-panel-btn');
    const closeRadarPanelBtn = document.getElementById('close-radar-panel-btn');
    const radarPanelOverlay = document.getElementById('radar-panel-overlay');
    const radarPanel = document.getElementById('radar-panel');
    const radarPanelContent = document.getElementById('radar-panel-content');

    // --- MODAL DE CONFIRMACIÓN (NUEVO) ---
    const modal = document.getElementById('confirmation-modal');
    const modalBox = modal.querySelector('div:first-of-type');
    const modalIcon = document.getElementById('modal-icon');
    const modalTitle = document.getElementById('modal-title');
    const modalMessage = document.getElementById('modal-message');
    const modalCancelBtn = document.getElementById('modal-cancel-btn');
    const modalConfirmBtn = document.getElementById('modal-confirm-btn');
    let onConfirmCallback = null;

    // --- LÓGICA DEL MODAL (NUEVA) ---
    function openModal({ title, message, icon, confirmText, confirmClass, onConfirm }) {
        modalTitle.textContent = title;
        modalMessage.innerHTML = message;
        modalIcon.setAttribute('data-lucide', icon);
        lucide.createIcons({ nodes: [modalIcon] });
        modalConfirmBtn.textContent = confirmText || 'CONFIRMAR';
        // Reset classes
        modalConfirmBtn.className = 'btn font-black py-2.5 px-4 rounded-lg uppercase w-full sm:w-auto flex items-center justify-center gap-2 shadow-md transition-all duration-200';
        
        if (confirmClass === 'danger') {
            modalConfirmBtn.classList.add('btn-primary'); // Usamos btn-primary (rojo) para peligro
        } else if (confirmClass === 'primary') {
            modalConfirmBtn.classList.add('btn-secondary'); // Usamos secundario (azul) para acciones normales si se requiere
        } else {
             modalConfirmBtn.classList.add('btn-primary');
        }
        
        onConfirmCallback = onConfirm; 
        modal.classList.remove('hidden');
        setTimeout(() => modalBox.classList.remove('scale-95', 'opacity-0'), 50);
        document.body.style.overflow = 'hidden';
    }

    function closeModal() {
        modalBox.classList.add('scale-95', 'opacity-0');
        setTimeout(() => {
            modal.classList.add('hidden');
            document.body.style.overflow = '';
        }, 300);
        onConfirmCallback = null;
    }

    modalCancelBtn.addEventListener('click', closeModal);
    modalConfirmBtn.addEventListener('click', () => {
        if (typeof onConfirmCallback === 'function') {
            onConfirmCallback();
        }
        closeModal();
    });

    // --- LÓGICA DEL PANEL LATERAL (NUEVA) ---
    function openRadarPanel() {
        radarPanel.classList.remove('translate-x-full');
        radarPanelOverlay.classList.remove('hidden');
        drawRadarPanelContent(radarGuardadoActual); 
    }

    function closeRadarPanel() {
        radarPanel.classList.add('translate-x-full');
        radarPanelOverlay.classList.add('hidden');
    }

    if(openRadarPanelBtn) openRadarPanelBtn.addEventListener('click', openRadarPanel);
    if(closeRadarPanelBtn) closeRadarPanelBtn.addEventListener('click', closeRadarPanel);
    if(radarPanelOverlay) radarPanelOverlay.addEventListener('click', closeRadarPanel);

    function drawRadarPanelContent(radarData) {
        if (radarData && radarData.puntuacion) {
            const formattedHtml = formatRadarResult(radarData); 
            radarPanelContent.innerHTML = `
                <div class="space-y-3">
                    <div class="flex justify-end mb-4 pr-3">
                        <button id="panel-radar-btn-eliminar" class="btn btn-primary text-white py-2 px-3 text-xs">
                            <i data-lucide="trash-2" class="w-4 h-4"></i> ELIMINAR
                        </button>
                    </div>
                    <div id="radar-display-container" class="ai-response p-4 border rounded-lg">
                        ${formattedHtml}
                    </div>
                </div>
            `;
            lucide.createIcons();
            assignRadarPanelButtonEvents();
        } else {
            radarPanelContent.innerHTML = `
                <div class="text-center p-4 border border-dashed rounded-lg">
                    <i data-lucide="info" class="w-12 h-12 text-gray-400 mx-auto mb-3"></i>
                    <p class="text-gray-500">
                        Aún no tienes un Radar de Diagnóstico guardado. ¡Genera uno para empezar!
                    </p>
                </div>
            `;
            lucide.createIcons();
        }
    }

    function assignRadarPanelButtonEvents() {
        const btnEliminar = document.getElementById('panel-radar-btn-eliminar');
        if (btnEliminar) {
            btnEliminar.addEventListener('click', () => {
                openModal({
                    title: 'CONFIRMAR ELIMINACIÓN',
                    message: '¿Estás seguro de que quieres eliminar tu Radar de Diagnóstico? <br> Esta acción no se puede deshacer.',
                    icon: 'alert-triangle',
                    confirmText: 'SÍ, ELIMINAR',
                    confirmClass: 'danger',
                    onConfirm: async () => {
                        await deleteRadarFromDB();
                        drawRadarPanelContent(radarGuardadoActual); 
                    }
                });
            });
        }
    }
    
    // --- LÓGICA CRUD (NUEVA) ---
    const radarEndpoints = {
        create: 'api-db/ia-business-diagnostic-radar-create.php',
        update: 'api-db/ia-business-diagnostic-radar-update.php',
        delete: 'api-db/ia-business-diagnostic-radar-delete.php'
    };
    
    async function createRadarInDB(radarData) {
        setProcessingState(true, 'Guardando...');
        try {
            const response = await fetch(radarEndpoints.create, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(radarData)
            });
            const result = await response.json();
            if (response.ok && result.success) {
                if (typeof showToast === 'function') showToast(result.message, 'success');
                radarGuardadoActual = radarData; // Actualiza la variable global
                openRadarPanel();
            } else {
                if (response.status === 409) {
                    throw new Error('A diagnostic radar already exists'); 
                }
                throw new Error(result.message || 'Error desconocido al crear Radar');
            }
        } catch (error) {
            throw error;
        } finally {
            setProcessingState(false);
        }
    }

    async function updateRadarInDB(radarData) {
        setProcessingState(true, 'Actualizando...');
        try {
            const response = await fetch(radarEndpoints.update, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(radarData)
            });
            const result = await response.json();
            if (response.ok && result.success) {
                if (typeof showToast === 'function') showToast(result.message, 'success');
                radarGuardadoActual = radarData;
            } else {
                throw new Error(result.message || 'Error desconocido al actualizar Radar');
            }
        } catch (error) {
            if (typeof showToast === 'function') showToast(`Error al actualizar Radar: ${error.message}`, 'error');
        } finally {
            setProcessingState(false);
        }
    }

    async function deleteRadarFromDB() {
        setProcessingState(true, 'Eliminando...');
        try {
            const response = await fetch(radarEndpoints.delete, {
                method: 'POST', 
                headers: { 'Content-Type': 'application/json' }
            });
            const result = await response.json();
            if (response.ok && result.success) {
                if (typeof showToast === 'function') showToast(result.message, 'success');
                radarGuardadoActual = null; // Limpia la variable global
            } else {
                throw new Error(result.message || 'Error desconocido al eliminar Radar');
            }
        } catch (error) {
            if (typeof showToast === 'function') showToast(`Error al eliminar Radar: ${error.message}`, 'error');
        } finally {
            setProcessingState(false);
        }
    }

    // --- LÓGICA DE FORMULARIO (Existente, pero actualizada) ---
    const optionsContainer = document.getElementById('diagnostic-options');
    const diagnosticAxes = [
        {
            axis: 'OBJETIVOS Y ESTRATEGIAS',
            icon: 'crosshair',
            subpoints: [
                { id: 'sp1-1', text: 'Siento que mi negocio avanza sin un rumbo claro o metas concretas y definidas.' },
                { id: 'sp1-2', text: 'No tengo un plan de acción con prioridades definidas para alcanzar mis objetivos.' },
                { id: 'sp1-3', text: 'Me es difícil saber si mis estrategias de marketing realmente me están funcionando.' }
            ]
        },
        {
            axis: 'IDENTIDAD CORPORATIVA',
            icon: 'gem',
            subpoints: [
                { id: 'sp2-1', text: 'La imagen de mi negocio no refleja calidad y podría parecer poco profesional.' },
                { id: 'sp2-2', text: 'Mi comunicación visual es inconsistente, lo que debilita la confianza en mi marca.' },
                { id: 'sp2-3', text: 'Siento que mi imagen actual no me ayuda a atraer al tipo de cliente quiero alcanzar.' }
            ]
        },
        {
            axis: 'PRESENCIA<br />DIGITAL',
            icon: 'monitor-smartphone',
            subpoints: [
                { id: 'sp3-1', text: 'Mis redes sociales están desactualizados y su contenido no genera interés.' },
                { id: 'sp3-2', text: 'Mi página web no inspira la confianza necesaria para que me contacten.' },
                { id: 'sp3-3', text: 'Mi negocio no está registrado en directorios clave como Google Mi Negocio.' }
            ]
        },
        {
            axis: 'ESTRATEGIA PUBLICITARIA',
            icon: 'megaphone',
            subpoints: [
                { id: 'sp4-1', text: 'Mi negocio depende de las recomendaciones para conseguir nuevos clientes.' },
                { id: 'sp4-2', text: 'He gastado dinero en publicidad, pero fue como tirar el dinero sin saber qué resultados trajo.' },
                { id: 'sp4-3', text: 'Cuando la publicidad atrae interesados, muchos de ellos se pierden no se convierten en clientes.' }
            ]
        },
        {
            axis: 'CANALES DE COMUNICACIÓN',
            icon: 'message-square',
            subpoints: [
                { id: 'sp5-1', text: 'Siento que pierdo tiempo y oportunidades al no saber en qué canales de comunicación enfocarme.' },
                { id: 'sp5-2', text: 'El desorden en mis canales de comunicación hace que dé una imagen poco profesional y pierda ventas.' },
                { id: 'sp5-3', text: 'No utilizo herramientas como WhatsApp Business para profesionalizar mi comunicación.' }
            ]
        },
        {
            axis: 'PROCESOS DE CONVERSIÓN',
            icon: 'filter',
            subpoints: [
                { id: 'sp6-1', text: 'Recibo muchas preguntas y solicitudes, pero un bajo porcentaje de ellas se convierten en clientes.' },
                { id: 'sp6-2', text: 'No tengo un proceso definido para dar seguimiento a los prospectos interesados.' },
                { id: 'sp6-3', text: 'No califico ni organizo a mis prospectos según su nivel de interés para priorizar mis esfuerzos.' }
            ]
        },
        {
            axis: 'SEGUIMIENTO<br />POST-VENTA',
            icon: 'handshake',
            subpoints: [
                { id: 'sp7-1', text: 'No aprovecho a mis clientes felices para que sus reseñas me ayuden a conseguir nuevos clientes.' },
                { id: 'sp7-2', text: 'No tengo un proceso para pedir a mis clientes actuales que me refieran con nuevos clientes.' },
                { id: 'sp7-3', text: 'No mantengo el contacto con mis clientes después de la venta para fomentar la lealtad.' }
            ]
        },
        {
            axis: 'MONITOREO Y EVALUACIÓN',
            icon: 'line-chart',
            subpoints: [
                { id: 'sp8-1', text: 'No sé qué canales o estrategias de marketing me están trayendo más clientes.' },
                { id: 'sp8-2', text: 'Invierto en crecer mi negocio, pero no sé si esas inversiones son rentables.' },
                { id: 'sp8-3', text: 'Tomo decisiones de negocio basadas en la intuición en lugar de datos y métricas reales.' }
            ]
        }
    ];
    
    optionsContainer.innerHTML = '';
    diagnosticAxes.forEach(axis => {
        const axisDiv = document.createElement('div');
        axisDiv.className = 'diagnostic-card'; // Clase CSS maestra
        let subpointsHtml = axis.subpoints.map(sp => {
            const isChecked = radarGuardadoActual && radarGuardadoActual.situations && radarGuardadoActual.situations.includes(`${axis.axis}: ${sp.text}`);
            return `
                <div class="subpoint-item">
                    <input type="checkbox" id="${sp.id}" name="${axis.axis}: ${sp.text}" ${isChecked ? 'checked' : ''}>
                    <label for="${sp.id}">${sp.text}</label>
                </div>
            `;
        }).join('');
        axisDiv.innerHTML = `
            <div class="axis-header">
                <i data-lucide="${axis.icon}" class="axis-icon"></i>
                <h3 class="axis-title">${axis.axis}</h3>
            </div>
            <div class="space-y-2">${subpointsHtml}</div>
        `;
        optionsContainer.appendChild(axisDiv);
    });
    lucide.createIcons();
    optionsContainer.addEventListener('click', e => {
        if(e.target.tagName === 'LABEL') {
            const checkbox = document.getElementById(e.target.getAttribute('for'));
            if(checkbox) checkbox.checked = !checkbox.checked;
        }
    });

    const setProcessingState = (isProcessing, customMessage = null) => {
        const btnText = document.getElementById('btn-text');
        const loaderContainer = document.getElementById('loader-container');
        
        loaderContainer.classList.toggle('hidden', !isProcessing);
        generateBtn.disabled = isProcessing;
        generateBtn.classList.toggle('opacity-50', isProcessing);
        generateBtn.classList.toggle('cursor-not-allowed', isProcessing);
        
        if (isProcessing) {
            loaderContainer.innerHTML = '<div class="w-6 h-6 border-4 border-t-transparent border-[var(--color-primary)] rounded-full animate-spin"></div>';
            if (customMessage) {
                btnText.textContent = customMessage;
            } else {
                btnText.textContent = 'Analizando Tu Situación...';
            }
        } else {
            btnText.textContent = 'GENERAR CON IA MARKETING';
        }
    };
    
    // --- FUNCIÓN DE RENDERIZADO (ACTUALIZADA) ---
    function formatRadarResult(data) {
        const planHtml = data.plan_de_accion.map((step, index) => `
            <div class="plan-action-step-item">
                <div class="plan-action-number-circle">${index + 1}</div>
                <div class="plan-action-step-text">
                    ${step.replace(/\*\*(.*?)\*\*/g, '<strong class="text-[var(--color-primary)]">$1</strong>')}
                </div>
            </div>`).join('');
            
        return `
            <div class="ai-response p-6 md:p-8 rounded-xl shadow-md space-y-8">
                <div class="text-center bg-gray-50 p-6 rounded-lg">
                    <h2 class="text-4xl font-black uppercase text-[var(--color-primary)] mb-3">Tu Puntuación <br />De Optimización Digital</h2>
                    <p class="text-6xl font-black text-brand-dark">${data.puntuacion}<span class="text-4xl">/100</span></p>
                </div>
                <div class="pt-8 border-t">
                    <h3 class="text-2xl font-bold text-gray-800 mb-4 flex items-center"><i data-lucide="search-check" class="mr-3 text-[var(--color-primary)]"></i>Diagnóstico General</h3>
                    <p class="text-gray-700 leading-relaxed">${data.diagnostico_general}</p>
                </div>
                <div class="pt-8 border-t">
                    <h3 class="text-2xl font-bold text-gray-800 mb-4 flex items-center"><i data-lucide="list-ordered" class="mr-3 text-[var(--color-secondary)]"></i>Tu Plan De Acción</h3>
                    <div class="mt-6 space-y-6">${planHtml}</div>
                </div>
            </div>`;
    }

    // --- FUNCIÓN DE VISUALIZACIÓN (ACTUALIZADA) ---
    function displayResult(data, showSaveButton = false) {
        let html = formatRadarResult(data);
        
        if (showSaveButton) {
            html += `
            <div id="action-buttons-container" class="mt-4 text-center">
                <button id="save-radar-btn" class="btn btn-primary py-3 px-6 rounded-lg flex items-center justify-center mx-auto mt-4 gap-2 hover:scale-105 transition-transform uppercase">
                    <i data-lucide="save"></i> <span>Guardar Diagnóstico</span>
                </button>
            </div>`;
        }
        
        resultContainer.innerHTML = html;
        lucide.createIcons();
        resultContainer.style.opacity = '1';
        resultContainer.scrollIntoView({ behavior: 'smooth', block: 'start' });
        
        if (showSaveButton) {
            document.getElementById('save-radar-btn').addEventListener('click', handleSaveRadar);
        }
    }
    
    // --- LÓGICA DE GUARDADO ---
    function handleSaveRadar() {
        if (!radarGeneradoReciente) {
            if (typeof showToast === 'function') showToast("No hay un diagnóstico reciente para guardar.", "error");
            return;
        }

        const saveAction = async () => {
            try {
                await createRadarInDB(radarGeneradoReciente);
            } catch (error) {
                if (error.message.includes('A diagnostic radar already exists')) {
                    // Si ya existe, pedir confirmación para sobrescribir
                    openModal({
                        title: 'CONFIRMAR SOBRESCRIBIR',
                        message: 'Ya tienes un Radar guardado. ¿Deseas reemplazarlo con la versión actual?',
                        icon: 'refresh-cw',
                        confirmText: 'SÍ, REEMPLAZAR',
                        confirmClass: 'danger',
                        onConfirm: () => {
                            updateRadarInDB(radarGeneradoReciente);
                            openRadarPanel();
                        }
                    });
                } else {
                    if (typeof showToast === 'function') showToast(`Error al guardar: ${error.message}`, "error");
                }
            }
        };
        saveAction();
    }
    
    // --- LÓGICA DE ENVÍO DE FORMULARIO ---
    form.addEventListener('submit', async function(event) {
        event.preventDefault();
        setProcessingState(true);
        
        resultContainer.innerHTML = '';
        resultContainer.style.opacity = '0';
        
        const selectedOptions = Array.from(form.querySelectorAll('input[type=checkbox]:checked')).map(cb => cb.name);
        const businessDescription = document.getElementById('diag-other').value.trim();
        const valueProposition = form.querySelector('input[name="value_proposition"]').value; 
        
        if (selectedOptions.length === 0 && !businessDescription) {
            if (typeof showToast === 'function') showToast('Por favor, selecciona al menos una opción o describe tu negocio.', 'error');
            setProcessingState(false);
            return;
        }
        
        try {
            const response = await fetch('api/api-business-diagnostic-radar.php', { 
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ 
                    situations: selectedOptions, 
                    business_description: businessDescription,
                    value_proposition: valueProposition 
                })
            });
            const data = await response.json();
            
            if (!response.ok || data.error) {
                let errorMsg = data.error || `El servidor respondió con un error ${response.status}.`;
                if(data.api_response_body && data.api_response_body.error) {
                    errorMsg = `Error de Google API: ${data.api_response_body.error.message}`;
                } else if (data.curl_error) {
                    errorMsg = `Error de cURL: ${data.curl_error}`;
                }
                throw new Error(errorMsg);
            }
            
            radarGeneradoReciente = data; // Guardar el resultado para el botón "Guardar"
            displayResult(data, true); // true = mostrar botón de guardar

        } catch (error) {
            console.error("Error:", error);
            resultContainer.innerHTML = `<div class="bg-[var(--color-secondary)] text-center p-4 text-white rounded-lg border border-[var(--color-secondary)]"><p><strong>Error:</strong> No se pudo generar el diagnóstico.</p><p class="mt-2 text-sm">${error.message || 'Error desconocido'}</p></div>`;
            resultContainer.style.opacity = '1';
        } finally {
            setProcessingState(false);
        }
    });
});
</script>
<script src="files/toast.js"></script>
</body>
</html>