<?php
declare(strict_types=1);

ob_start();

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/../core/api.php';
global $pdo;

ini_set('display_errors', '0');
ini_set('log_errors', '1');
error_reporting(E_ALL);

header('Content-Type: application/json');

$response = ['success' => false, 'data' => [], 'message' => 'ERROR DESCONOCIDO'];

try {
    $userId = $_SESSION['user_id'] ?? 1; 
    $today = date('Y-m-d');
    $data = [];

    // HELPERS
    function sanitize_output(?string $value): ?string {
        return $value !== null ? htmlspecialchars($value, ENT_QUOTES, 'UTF-8') : null;
    }

    // FIX: Se usan claves en inglés para los datos mensuales
    function process_monthly_data(array &$monthly_data_ref, array $db_rows, array $metrics_map): void {
        foreach ($db_rows as $row) {
            $month_key = $row['month_key'];
            if (!isset($monthly_data_ref[$month_key])) {
                $monthly_data_ref[$month_key] = [];
            }
            foreach ($metrics_map as $db_col => $data_key) {
                if (isset($row[$db_col])) {
                    $monthly_data_ref[$month_key][$data_key] = ($data_key === 'totalRevenue') 
                        ? (float)$row[$db_col] 
                        : (int)$row[$db_col];
                }
            }
        }
    }

    // --- CONFIGURACIÓN DE ESTADOS (BASE DE DATOS EN INGLÉS) ---
    $tracking_statuses = ['New', 'Contacted', 'Qualified', 'Proposal Sent', 'Negotiation'];
    $overdue_invoice_statuses = ['sent', 'pending', 'overdue']; 
    $pending_quote_statuses = ['generated', 'sent']; 

    $tracking_placeholders = implode(',', array_fill(0, count($tracking_statuses), '?'));
    $overdue_placeholders = implode(',', array_fill(0, count($overdue_invoice_statuses), '?'));
    $quote_placeholders = implode(',', array_fill(0, count($pending_quote_statuses), '?'));

    // 1. CONTEOS GENERALES (FIX: pendingTasks cuenta TODAS las pendientes para igualar la UI)
    $sql_counts = "
        SELECT
            (SELECT COUNT(id) FROM unique_appointments WHERE user_id = ? AND status IN ('pending', 'confirmed') AND appointment_date >= ?) AS pendingAppointments,
            (SELECT COUNT(id) FROM leads WHERE user_id = ? AND status IN ($tracking_placeholders)) AS trackingProspects,
            (SELECT COUNT(id) FROM tasks WHERE user_id = ? AND status = 'pending') AS pendingTasks,
            (SELECT COUNT(id) FROM quotes WHERE user_id = ? AND status IN ($quote_placeholders)) AS pendingEstimates,
            (SELECT COUNT(id) FROM invoices WHERE user_id = ? AND status IN ($overdue_placeholders) AND (due_date IS NOT NULL AND due_date <= ?)) AS overdueInvoices
    ";

    $stmt_counts = $pdo->prepare($sql_counts);

    $params = [$userId, $today]; 
    $params[] = $userId; $params = array_merge($params, $tracking_statuses); 
    $params[] = $userId; 
    $params[] = $userId; $params = array_merge($params, $pending_quote_statuses); 
    $params[] = $userId; $params = array_merge($params, $overdue_invoice_statuses); $params[] = $today;

    $stmt_counts->execute($params);
    $counts = $stmt_counts->fetch(PDO::FETCH_ASSOC);

    $data = $counts ?: [
        'pendingAppointments' => 0, 'trackingProspects' => 0, 
        'pendingTasks' => 0, 'pendingEstimates' => 0, 'overdueInvoices' => 0
    ];

    // 2. CITAS PRÓXIMAS (FIX: Texto del servicio traducido a inglés en el IFNULL)
    $stmt_appointments = $pdo->prepare("
        SELECT 
            ua.appointment_date AS fecha, 
            ua.appointment_time AS hora, 
            ua.status, 
            COALESCE(
                NULLIF(TRIM(CONCAT_WS(' ', c.first_name, c.last_name)), ''), 
                NULLIF(TRIM(CONCAT_WS(' ', l.first_name, l.last_name)), ''), 
                'Cita sin nombre'
            ) AS associated_name, 
            IFNULL(ua.notes, 'Service Not Specified') AS servicio 
        FROM unique_appointments ua 
        LEFT JOIN clients c ON ua.client_id = c.id 
        LEFT JOIN leads l ON ua.lead_id = l.id 
        WHERE ua.user_id = ? 
          AND ua.status = 'confirmed' 
          AND ua.appointment_date >= ? 
        ORDER BY ua.appointment_date ASC, ua.appointment_time ASC 
        LIMIT 5
    ");
    $stmt_appointments->execute([$userId, $today]);
    
    $data['upcomingAppointments'] = [];
    while ($row = $stmt_appointments->fetch(PDO::FETCH_ASSOC)) {
        $row['associated_name'] = sanitize_output($row['associated_name']);
        $row['servicio'] = sanitize_output($row['servicio']);
        $data['upcomingAppointments'][] = $row;
    }

    // 3. PROSPECTOS EN SEGUIMIENTO (FIX: Mapeo de status_seguimiento para JS)
    $stmt_prospects = $pdo->prepare("
        SELECT id, first_name, last_name, email, phone, mobile, status, status AS status_seguimiento, created_at 
        FROM leads 
        WHERE user_id = ? AND status IN ($tracking_placeholders) 
        ORDER BY created_at DESC 
        LIMIT 5
    ");
    $stmt_prospects->execute(array_merge([$userId], $tracking_statuses));
    
    $data['trackingProspectsList'] = [];
    while ($row = $stmt_prospects->fetch(PDO::FETCH_ASSOC)) {
        $data['trackingProspectsList'][] = [
            'id' => $row['id'],
            'first_name' => sanitize_output($row['first_name']),
            'last_name' => sanitize_output($row['last_name']),
            'email' => sanitize_output($row['email']),
            'phone' => sanitize_output($row['phone']),
            'mobile' => sanitize_output($row['mobile']),
            'estado' => sanitize_output($row['status']), 
            'status_seguimiento' => sanitize_output($row['status_seguimiento']), 
            'created_at' => $row['created_at']
        ];
    }

    // 4. ESTIMADOS/QUOTES PENDIENTES (FIX: Mapeo estricto de claves para JS)
    $stmt_quotes = $pdo->prepare("
        SELECT 
            q.external_id AS id, 
            q.total_amount, 
            q.estimate_date, 
            q.status, 
            COALESCE(
                NULLIF(TRIM(CONCAT_WS(' ', c.first_name, c.last_name)), ''), 
                NULLIF(TRIM(CONCAT_WS(' ', l.first_name, l.last_name)), ''), 
                'Estimado sin nombre'
            ) AS client_name 
        FROM quotes q 
        LEFT JOIN clients c ON q.client_id = c.id 
        LEFT JOIN leads l ON q.lead_id = l.id 
        WHERE q.user_id = ? AND q.status IN ($quote_placeholders) 
        ORDER BY q.estimate_date ASC 
        LIMIT 5
    ");
    $stmt_quotes->execute(array_merge([$userId], $pending_quote_statuses));
    
    $data['pendingEstimatesList'] = [];
    while ($row = $stmt_quotes->fetch(PDO::FETCH_ASSOC)) {
        $data['pendingEstimatesList'][] = [
            'id' => $row['id'], 
            'total_amount' => $row['total_amount'],
            'estimate_date' => $row['estimate_date'],
            'status' => sanitize_output($row['status']),
            'client_name' => sanitize_output($row['client_name']), 
            'lead_name' => null, 
            'external_id' => $row['id']
        ];
    }

    // 5. FACTURAS VENCIDAS (FIX: Mapeo estricto de claves para JS)
    $stmt_invoices = $pdo->prepare("
        SELECT 
            i.external_id AS id, 
            i.total_amount, 
            i.due_date,
            i.status,
            COALESCE(
                NULLIF(TRIM(CONCAT_WS(' ', c.first_name, c.last_name)), ''), 
                NULLIF(TRIM(CONCAT_WS(' ', l.first_name, l.last_name)), ''), 
                'Factura sin nombre'
            ) AS client_name
        FROM invoices i
        LEFT JOIN clients c ON i.client_id = c.id
        LEFT JOIN leads l ON i.lead_id = l.id
        WHERE i.user_id = ? 
          AND i.status IN ($overdue_placeholders) 
          AND (i.status = 'overdue' OR (i.due_date IS NOT NULL AND i.due_date <= ?))
        ORDER BY i.due_date ASC
        LIMIT 5
    ");
    $stmt_invoices->execute(array_merge([$userId], $overdue_invoice_statuses, [$today]));
    
    $data['overdueInvoicesList'] = [];
    while ($row = $stmt_invoices->fetch(PDO::FETCH_ASSOC)) {
        $data['overdueInvoicesList'][] = [
            'id' => $row['id'], 
            'total_amount' => $row['total_amount'],
            'due_date' => $row['due_date'],
            'status' => sanitize_output($row['status']),
            'client_name' => sanitize_output($row['client_name']), 
            'lead_name' => null, 
            'external_id' => $row['id']
        ];
    }

    // 6. TODAS LAS TAREAS
    $stmt_tasks = $pdo->prepare("
        SELECT t.*, 
            COALESCE(TRIM(CONCAT_WS(' ', c.first_name, c.last_name)), c.email, c.phone) AS client_name, 
            COALESCE(TRIM(CONCAT_WS(' ', l.first_name, l.last_name)), l.email, l.phone) AS lead_name 
        FROM tasks t 
        LEFT JOIN clients c ON t.client_id = c.id 
        LEFT JOIN leads l ON t.lead_id = l.id 
        WHERE t.user_id = ?
        ORDER BY t.status ASC, t.due_date ASC
    ");
    $stmt_tasks->execute([$userId]);
    
    $data['allTasks'] = [];
    while ($row = $stmt_tasks->fetch(PDO::FETCH_ASSOC)) {
        $clean_row = [];
        foreach ($row as $key => $val) {
            $clean_row[$key] = sanitize_output((string)$val);
        }
        $data['allTasks'][] = $clean_row;
    }

    // 7. DATOS MENSUALES (FIX: Claves en inglés)
    $data['monthly_data'] = [];

    // Leads
    $stmt_m_leads = $pdo->prepare("SELECT DATE_FORMAT(created_at, '%Y-%m') AS month_key, COUNT(id) AS count FROM leads WHERE user_id = ? GROUP BY month_key");
    $stmt_m_leads->execute([$userId]);
    process_monthly_data($data['monthly_data'], $stmt_m_leads->fetchAll(PDO::FETCH_ASSOC), ['count' => 'leads']);

    // Clientes
    $stmt_m_clients = $pdo->prepare("SELECT DATE_FORMAT(created_at, '%Y-%m') AS month_key, COUNT(id) AS count FROM clients WHERE user_id = ? GROUP BY month_key");
    $stmt_m_clients->execute([$userId]);
    process_monthly_data($data['monthly_data'], $stmt_m_clients->fetchAll(PDO::FETCH_ASSOC), ['count' => 'clients']);

    // Facturación y Servicios 
    $stmt_m_fin = $pdo->prepare("
        SELECT 
            DATE_FORMAT(invoice_date, '%Y-%m') AS month_key, 
            SUM(total_amount) AS total_sum, 
            COUNT(id) AS count 
        FROM invoices 
        WHERE user_id = ? AND status IN ('paid') 
        GROUP BY month_key
    ");
    $stmt_m_fin->execute([$userId]);
    process_monthly_data($data['monthly_data'], $stmt_m_fin->fetchAll(PDO::FETCH_ASSOC), ['total_sum' => 'totalRevenue', 'count' => 'paidInvoices']);

    // Servicios Vendidos
    $stmt_m_serv = $pdo->prepare("
        SELECT DATE_FORMAT(i.invoice_date, '%Y-%m') AS month_key, COUNT(ii.id) AS services_count 
        FROM invoices i 
        JOIN invoice_items ii ON i.id = ii.invoice_id 
        WHERE i.user_id = ? AND i.status IN ('paid') 
        GROUP BY month_key
    ");
    $stmt_m_serv->execute([$userId]);
    process_monthly_data($data['monthly_data'], $stmt_m_serv->fetchAll(PDO::FETCH_ASSOC), ['services_count' => 'servicesSold']);

    $response['success'] = true;
    $response['data'] = $data;
    $response['message'] = 'Datos cargados correctamente';

} catch (PDOException $e) {
    error_log("DB Error (dashboard-read): " . $e->getMessage());
    $response['message'] = 'Error al cargar los datos del tablero.';
} catch (Exception $e) {
    error_log("General Error (dashboard-read): " . $e->getMessage());
    $response['message'] = $e->getMessage();
}

if (ob_get_level() > 0) {
    ob_end_clean();
}
echo json_encode($response);
exit;