<?php
declare(strict_types=1);

// 1. CARGA DEL ENTORNO API (Maneja cabeceras JSON, sesin y conexin $pdo)
// Esto tambin incluye core/init.php, haciendo disponible $user_id_session.
require_once __DIR__ . '/../core/api.php';
require_once __DIR__ . '/../core/security.php';

// CRITICAL FIX: Definir la variable de sesin para el contexto API, ya que core/api.php no lo hace.
$user_id_session = filter_var($_SESSION['user_id'] ?? null, FILTER_VALIDATE_INT) ?: null;

$userId = $user_id_session;

try {
    if (!$userId) {
        throw new Exception('SESSION EXPIRED');
    }

    // 2. RECOLECCI07N DE DATOS (Maneja tanto POST como JSON input)
    // Se utiliza la funcin de sanitizacin de core/security.php
    $data = sanitizeData($_POST);
    if (empty($data)) {
        $data = readJsonInput(); // Usamos la funcin de core/security.php
        $data = sanitizeData($data); // Se sanitiza el input JSON
    }

    $action = $data['action'] ?? '';
    $projectId = filter_var($data['id'] ?? 0, FILTER_VALIDATE_INT);

    if (!$projectId) {
        throw new Exception('INVALID PROJECT ID');
    }

    // 3. VERIFICAR PROPIEDAD (Usa el user_id para validar el permiso)
    $stmtCheck = $pdo->prepare("SELECT id FROM projects WHERE id = ? AND user_id = ?");
    $stmtCheck->execute([$projectId, $userId]);
    
    if ($stmtCheck->rowCount() === 0) {
        throw new Exception('PERMISSION DENIED TO EDIT THIS PROJECT');
    }

    // 4. L07GICA DE ACTUALIZACI07N
    if ($action === 'edit_details') {
        if (empty($data['name']) || empty($data['client_id']) || empty($data['service_id']) || empty($data['start_date'])) {
            throw new Exception('MISSING REQUIRED DATA');
        }

        $name = $data['name'];
        $client_id = filter_var($data['client_id'], FILTER_VALIDATE_INT);
        $service_id = filter_var($data['service_id'], FILTER_VALIDATE_INT);
        $start_date = $data['start_date'];
        $due_date = !empty($data['due_date']) ? $data['due_date'] : null;
        $description = $data['description'] ?? '';

        // Se verifica que el cliente sea del usuario (seguridad doble)
        $checkItems = $pdo->prepare("SELECT COUNT(*) as client_ok FROM clients WHERE id = ? AND user_id = ?");
        $checkItems->execute([$client_id, $userId]);
        $resCheck = $checkItems->fetch(PDO::FETCH_ASSOC);

        if ($resCheck['client_ok'] == 0) {
            throw new Exception('INVALID CLIENT ID');
        }

        $sql = "UPDATE projects SET name = ?, client_id = ?, service_id = ?, start_date = ?, due_date = ?, description = ?, updated_at = NOW() WHERE id = ? AND user_id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$name, $client_id, $service_id, $start_date, $due_date, $description, $projectId, $userId]);

        $response = ['success' => true, 'message' => 'PROYECTO ACTUALIZADO'];

    } elseif ($action === 'update_status') {
        $newStatus = $data['status'] ?? '';
        
        // El valor de los datos DEBE estar en espa09ol para coincidir con el frontend (projects.php)
        $allowed_statuses = ['Aprobado', 'En Planificacin', 'En Proceso', 'Listo para Revisin', 'Esperando Aprobacin', 'Completado', 'Archivado'];
        
        if (!in_array($newStatus, $allowed_statuses)) {
            // Lgica en ingls: el valor del dato (status) es invlido
            throw new Exception('INVALID STATUS VALUE: ' . $newStatus);
        }

        // Se usa Prepared Statement y se valida el user_id
        $sql = "UPDATE projects SET status = ?, updated_at = NOW() WHERE id = ? AND user_id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$newStatus, $projectId, $userId]);

        $response = ['success' => true, 'message' => 'ESTADO ACTUALIZADO'];
    } else {
        throw new Exception('UNRECOGNIZED ACTION');
    }

    // 5. RESPUESTA CON DATOS FRESCOS PARA EL FRONTEND
    if ($response['success']) {
        $stmtGet = $pdo->prepare("
            SELECT p.id, p.name, p.description, p.start_date, p.due_date, p.status, p.updated_at, 
                   c.first_name, c.last_name, p.client_id, 
                   s.name as service_name, s.id as service_id
            FROM projects p
            LEFT JOIN clients c ON p.client_id = c.id
            LEFT JOIN services s ON p.service_id = s.id
            WHERE p.id = ? AND p.user_id = ?
        ");
        $stmtGet->execute([$projectId, $userId]);
        $project = $stmtGet->fetch(PDO::FETCH_ASSOC);

        if ($project) {
            $project['client_name'] = trim(($project['first_name'] ?? 'N/A') . ' ' . ($project['last_name'] ?? ''));
            // Sanitizacin antes de devolver al frontend
            $project['name'] = htmlspecialchars($project['name'], ENT_QUOTES, 'UTF-8');
            $project['description'] = htmlspecialchars($project['description'] ?? '', ENT_QUOTES, 'UTF-8');
            unset($project['first_name'], $project['last_name']);
            $response['project'] = $project;
        }
    }
    
    // Funcin de core/api.php para limpieza de buffer y salida JSON
    sendJsonResponse($response);

} catch (Exception $e) {
    // Manejo de errores y salida JSON limpia
    error_log("Project Update Error: " . $e->getMessage()); 
    sendJsonResponse([
        'success' => false, 
        'message' => 'ERROR: ' . $e->getMessage()
    ], 400);
}